'use client';

import { useState, FormEvent, ChangeEvent } from 'react';
import { useTranslations } from 'next-intl';

interface FormData {
  firstName: string;
  lastName: string;
  email: string;
  phone: string;
  subject: string;
  format: string;
}

interface FormStatus {
  loading: boolean;
  success: boolean;
  error: string;
}

export default function InvestmentFormMobile() {
  const t = useTranslations('InvestorsPage');
  const [formData, setFormData] = useState<FormData>({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    subject: '',
    format: ''
  });

  const [status, setStatus] = useState<FormStatus>({
    loading: false,
    success: false,
    error: ''
  });

  const handleChange = (e: ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSubmit = async (e: FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setStatus({ loading: true, success: false, error: '' });

    try {
      const response = await fetch('/api/investment', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(formData),
      });

      if (!response.ok) {
        throw new Error('Failed to send investment request');
      }

      setStatus({ loading: false, success: true, error: '' });
      setFormData({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        subject: '',
        format: ''
      });
    } catch (error) {
      setStatus({ loading: false, success: false, error: t('form_error_send_message_try_again') });
    }
  };

  return (
    <form onSubmit={handleSubmit} className="flex flex-col justify-center items-center gap-6 w-full">
      <div className="flex flex-col items-start gap-3 w-full">
        <div className="flex items-start gap-3 w-full">
          <input 
            type="text" 
            name="firstName"
            value={formData.firstName}
            onChange={handleChange}
            placeholder={t('form_fname')} 
            required
            className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] min-w-0"
          />
          <input 
            type="text" 
            name="lastName"
            value={formData.lastName}
            onChange={handleChange}
            placeholder={t('form_lname')}
            required
            className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] min-w-0"
          />
        </div>
        <div className="flex items-start gap-3 w-full">
          <input 
            type="email" 
            name="email"
            value={formData.email}
            onChange={handleChange}
            placeholder={t('form_email')}
            required
            className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] min-w-0"
          />
          <input 
            type="tel" 
            name="phone"
            value={formData.phone}
            onChange={handleChange}
            placeholder={t('form_phone')}
            required
            className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#888] font-roboto text-lg font-light leading-[150%] min-w-0"
          />
        </div>
        <div className="flex flex-col items-start gap-3 w-full">
          <input 
            type="text" 
            name="subject"
            value={formData.subject}
            onChange={handleChange}
            placeholder={t('form_subject')}
            required
            className="flex py-2 px-4 items-start gap-1.5 w-full bg-[#E6E6E6] text-[#888] font-roboto text-lg font-light leading-[150%]"
          />
          <input 
            type="text" 
            name="format"
            value={formData.format}
            onChange={handleChange}
            placeholder={t('form_format')}
            required
            className="flex py-2 px-4 items-start gap-1.5 w-full bg-[#E6E6E6] text-[#888] font-roboto text-lg font-light leading-[150%]"
          />
        </div>
      </div>
      
      {status.error && (
        <p className="text-red-500 text-sm w-full">{status.error}</p>
      )}
      
      {status.success && (
        <p className="text-green-500 text-sm w-full">{t('form_success_message')}</p>
      )}
      
      <button 
        type="submit"
        disabled={status.loading}
        className="flex py-3 px-6 justify-center items-center gap-2 w-full rounded-sm bg-[#A79D94] disabled:opacity-50"
      >
        <span className="text-white font-roboto text-lg font-normal leading-[150%]">
          {status.loading ? t('form_sending') : t('form_button')}
        </span>
      </button>
    </form>
  );
} 