'use client';

import { useEffect } from 'react';
import Image from 'next/image';

interface ImageModalProps {
  src: string;
  alt: string;
  onClose: () => void;
}

export default function ImageModal({ src, alt, onClose }: ImageModalProps) {
  useEffect(() => {
    const handleEscape = (e: KeyboardEvent) => {
      if (e.key === 'Escape') {
        onClose();
      }
    };

    document.addEventListener('keydown', handleEscape);
    document.body.style.overflow = 'hidden';

    return () => {
      document.removeEventListener('keydown', handleEscape);
      document.body.style.overflow = 'unset';
    };
  }, [onClose]);

  return (
    <div 
      role="button"
      tabIndex={0}
      className="fixed inset-0 bg-black bg-opacity-90 flex items-center justify-center z-50 p-4"
      onClick={(e) => {
        // Close modal if backdrop is clicked
        if (e.target === e.currentTarget) {
          onClose();
        }
      }}
      onKeyDown={(e) => {
        // Close modal if Enter is pressed on the backdrop
        if (e.key === 'Enter' && e.target === e.currentTarget) {
          onClose();
        }
      }}
    >
      <div 
        role="dialog"
        aria-modal="true"
        aria-label={alt}
        className="relative w-auto h-auto max-w-[90vw] max-h-[90vh]"
      >
        {/* Close button */}
        <button
          onClick={onClose}
          className="absolute -top-12 right-0 text-white text-2xl hover:text-gray-300 transition-colors z-10"
          aria-label="Close modal"
        >
          ✕
        </button>
        
        {/* Image */}
        <div className="relative w-full h-full">
          <Image
            src={src}
            alt={alt}
            layout="fill"
            objectFit="contain"
            className="rounded-lg shadow-2xl"
          />
        </div>
      </div>
    </div>
  );
} 