'use client';

import Image from 'next/image';
import Link from 'next/link';
import { useTranslations, useLocale } from 'next-intl';
import { useState, useEffect } from 'react';

export default function Footer() {
  const t = useTranslations('Footer');
  const locale = useLocale();
  const [isDesktop, setIsDesktop] = useState(false);
  const getLocalizedPath = (path: string) => `/${locale}${path}`;

  useEffect(() => {
    const checkScreenSize = () => {
      setIsDesktop(window.innerWidth >= 1024);
    };

    checkScreenSize();
    window.addEventListener('resize', checkScreenSize);

    return () => window.removeEventListener('resize', checkScreenSize);
  }, []);

  const [formData, setFormData] = useState({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    subject: '',
    message: ''
  });

  const [status, setStatus] = useState<{ loading: boolean; success: boolean; error: string }>({
    loading: false,
    success: false,
    error: ''
  });

  const socialIcons = [
    { name: "X (Twitter)", icon: "/images/logo/icon.svg", url: "https://x.com/LAKE_RES2" },
    { name: "BSCScan", icon: "/images/logo/icon2.svg", url: "https://bscscan.com/token/0xa5706783e577b1a9a9a89502efb5387fedaec6e0" },
    { name: "Discord", icon: "/images/logo/icon_discord.svg", bgColor: "#42382F", url: "https://discord.com/users/lake_resort_residence_86721" },
    { name: "YouTube", icon: "/images/logo/icon_youtube.svg", bgColor: "#42382F", url: "https://www.youtube.com/@LAKE_RESORT_RESIDENCE" },
    { name: "Telegram", icon: "/images/logo/icon_telegram.svg", bgColor: "#42382F", url: "https://t.me/lakeinvestorhub" },
    { name: "Instagram", icon: "/images/logo/icon_inst.svg", bgColor: "#42382F", url: "https://www.instagram.com/lake_resort_residence/" },
    { name: "Reddit", icon: "/images/logo/icon_reddit.svg", bgColor: "#42382F", url: "https://www.reddit.com/r/LAKERESORTRESIDENCE/hot/" },
    { name: "GitHub", icon: "/images/logo/icon_github.svg", bgColor: "#42382F", url: "https://github.com/LAKERESORTRESIDENCE/lake-token-whitepaper-md" }
  ];

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setStatus({ loading: true, success: false, error: '' });
    try {
      const response = await fetch('/api/contact', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(formData),
      });
      if (!response.ok) {
        throw new Error(t('form_error_send_message'));
      }
      setStatus({ loading: false, success: true, error: '' });
      setFormData({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        subject: '',
        message: ''
      });
    } catch (error) {
      setStatus({ loading: false, success: false, error: t('form_error_send_message_try_again') });
    }
  };

  if (isDesktop) {
    // Desktop version
    return (
      <footer id="footer-section" className="flex py-[60px] px-8 pb-4 flex-col items-center gap-2 self-stretch bg-[#A79D94]">
        <div className="flex max-w-[1376px] w-full flex-col items-start gap-2">
          {/* Contact Section */}
          <div className="flex flex-col items-start gap-8 self-stretch">
            <div className="flex justify-between items-start self-stretch gap-12">
              
              {/* Contact Methods */}
              <div className="flex flex-col justify-between items-start flex-1 min-w-0">
                <div className="flex flex-col items-start gap-8 self-stretch">
                  <div className="flex flex-col items-start gap-4 self-stretch">
                    <h3 className="self-stretch text-white font-roboto text-2xl font-light leading-[130%]">
                      {t('contact_title')}
                    </h3>
                    <div className="flex flex-col items-start gap-2 self-stretch">
                      <p className="self-stretch text-white font-roboto text-lg font-light leading-[150%]">
                        +996 223 660006<br/>
                        +996 221 111441
                      </p>
                      <p className="self-stretch text-white font-roboto text-lg font-light leading-[150%]">
                        info@lakeresortresidence.com
                      </p>
                    </div>
                  </div>
                  
                  {/* Social Media Icons */}
                  <div className="flex items-start content-start gap-2 flex-wrap">
                    {socialIcons.map((icon, index) => (
                      <Link 
                        key={index}
                        href={icon.url}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="w-10 h-10 flex-shrink-0 rounded-full flex justify-center items-center transition-opacity hover:opacity-80"
                        style={icon.bgColor ? { backgroundColor: icon.bgColor } : {}}
                      >
                        <Image
                          src={icon.icon}
                          alt={icon.name}
                          width={40}
                          height={40}
                          className="flex-shrink-0"
                          style={{ maxWidth: '100%', height: 'auto' }}
                        />
                      </Link>
                    ))}
                    {/* Social Media Group with text */}
                    <div className="flex items-end">
                      <Link
                        href="https://www.linkedin.com/groups/13198208/"
                        target="_blank"
                        rel="noopener noreferrer"
                        className="w-10 h-10 flex-shrink-0 rounded-full bg-[#42382F] flex justify-center items-center transition-opacity hover:opacity-80"
                      >
                        <Image
                          src="/images/logo/icon_linkedin.svg"
                          alt="LinkedIn Group"
                          width={40}
                          height={40}
                          className="flex-shrink-0"
                          style={{ maxWidth: '100%', height: 'auto' }}
                        />
                      </Link>
                      <span className="text-[#666] font-manrope text-sm font-normal leading-[150%] tracking-[-0.28px]">
                      </span>
                    </div>
                  </div>
                </div>
              </div>

              {/* Contact Form Section */}
              <div className="flex w-[601px] flex-col justify-center items-center gap-6">
                <h3 className="self-stretch text-white font-roboto text-2xl font-light leading-[130%]">
                  {t('contact_form_title')}
                </h3>
                
                <form onSubmit={handleSubmit} className="flex h-[255px] flex-col items-start gap-3 self-stretch">
                  {/* Name Fields */}
                  <div className="flex items-start gap-3 self-stretch">
                    <input
                      type="text"
                      name="firstName"
                      placeholder={t('form_fname')}
                      value={formData.firstName}
                      onChange={handleInputChange}
                      className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none"
                      required
                    />
                    <input
                      type="text"
                      name="lastName"
                      placeholder={t('form_lname')}
                      value={formData.lastName}
                      onChange={handleInputChange}
                      className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none"
                      required
                    />
                  </div>
                  
                  {/* Email and Phone Fields */}
                  <div className="flex items-start gap-3 self-stretch">
                    <input
                      type="email"
                      name="email"
                      placeholder={t('form_email')}
                      value={formData.email}
                      onChange={handleInputChange}
                      className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none"
                      required
                    />
                    <input
                      type="tel"
                      name="phone"
                      placeholder={t('form_phone')}
                      value={formData.phone}
                      onChange={handleInputChange}
                      className="flex py-2 px-4 items-start gap-1.5 flex-1 bg-[#E6E6E6] text-[#888] font-roboto text-lg font-light leading-[150%] border-none outline-none"
                      required
                    />
                  </div>
                  
                  {/* Subject Field */}
                  <input
                    type="text"
                    name="subject"
                    placeholder={t('form_subject')}
                    value={formData.subject}
                    onChange={handleInputChange}
                    className="flex py-2 px-4 items-start gap-1.5 self-stretch bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none"
                    required
                  />
                  
                  {/* Message Field */}
                  <textarea
                    name="message"
                    placeholder={t('form_message')}
                    value={formData.message}
                    onChange={handleInputChange}
                    className="flex py-2 px-4 items-start gap-1.5 flex-1 self-stretch bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none resize-none"
                    required
                  />
                  
                  {/* Submit Button */}
                  <button
                    type="submit"
                    disabled={status.loading}
                    className="flex h-12 py-3 px-6 justify-center items-center gap-2 self-stretch rounded-sm border border-[#A79D94] bg-white hover:bg-gray-50 transition-colors"
                  >
                    <span className="text-[#A79D94] font-roboto text-lg font-semibold leading-normal">
                      {status.loading ? t('form_sending') : t('form_submit')}
                    </span>
                  </button>
                  {status.error && (
                    <p className="text-red-500 text-sm">{status.error}</p>
                  )}
                  {status.success && (
                    <p className="text-green-500 text-sm">{t('form_success_message')}</p>
                  )}
                </form>
              </div>
            </div>
            
            {/* Spacer */}
            <div className="h-px self-stretch bg-[rgba(255,255,255,0.1)]"></div>
          </div>

          {/* Footer Bottom */}
          <div className="flex flex-col items-center gap-2 self-stretch">
            <p className="self-stretch text-white font-roboto text-lg font-light leading-[150%]">
              {t('license_info')}
            </p>
            
            <div className="flex justify-between items-center self-stretch">
              <div className="flex justify-center items-center gap-4">
                <div className="w-[65px] h-[65px] flex-shrink-0">
                  <Image
                    src="/images/logo/logotype_black.svg"
                    alt="LAKE Resort Residence"
                    width={65}
                    height={65}
                    className="w-full h-full object-contain"
                  />
                </div>
                <p className="text-white font-roboto text-lg font-light leading-[150%] max-w-[438px]" dangerouslySetInnerHTML={{ __html: t('copyright').replace('Все права защищены.', '<br/>Все права защищены.') }} />
              </div>
              
              <div className="flex items-center gap-6">
                <Link 
                  href={getLocalizedPath("/terms-of-use")} 
                  className="text-white font-roboto text-lg font-light leading-[150%] hover:opacity-80 transition-opacity"
                >
                  {t('terms_of_use')}
                </Link>
                <div className="w-0.5 self-stretch bg-[rgba(255,255,255,0.1)]"></div>
                <Link 
                  href={getLocalizedPath("/privacy-statement")} 
                  className="text-white font-roboto text-lg font-light leading-[150%] hover:opacity-80 transition-opacity"
                >
                  {t('privacy_statement')}
                </Link>
                <div className="w-0.5 self-stretch bg-[rgba(255,255,255,0.1)]"></div>
                <Link 
                  href={getLocalizedPath("/privacy-settings")} 
                  className="text-white font-roboto text-lg font-light leading-[150%] hover:opacity-80 transition-opacity"
                >
                  {t('privacy_settings')}
                </Link>
              </div>
            </div>
          </div>
        </div>
      </footer>
    );
  }

  // Mobile version
  return (
    <footer id="footer-section" className="flex px-3 pt-[60px] pb-4 flex-col items-center gap-8 self-stretch bg-[#A79D94]">
      {/* Contact Text - мобильная версия */}
      <div className="flex flex-col items-start gap-4 w-full max-w-[351px]">
        <h3 className="w-full text-white font-roboto text-2xl font-light leading-[130%]">
          {t('contact_title')}
        </h3>
        <div className="flex flex-col items-start gap-2 w-full">
          <p className="w-full text-white font-roboto text-lg font-light leading-[150%]">
            +996 223 660006<br/>
            +996 221 111441
          </p>
          <p className="w-full text-white font-roboto text-lg font-light leading-[150%]">
            info@lakeresortresidence.com
          </p>
        </div>
      </div>

      {/* Contact Form Section - мобильная версия */}
      <div className="flex flex-col justify-center items-center gap-6 w-full max-w-[351px] h-[413px]">
        <h3 className="w-full text-white font-roboto text-2xl font-light leading-[130%]">
          {t('contact_form_title')}
        </h3>
        
        <form onSubmit={handleSubmit} className="flex h-[255px] flex-col items-start gap-3 w-full">
          {/* Name Fields */}
          <div className="flex items-start gap-3 w-full h-[43px]">
            <input
              type="text"
              name="firstName"
              placeholder={t('form_fname')}
              value={formData.firstName}
              onChange={handleInputChange}
              className="flex-1 py-2 px-4 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none h-full min-w-0"
              required
            />
            <input
              type="text"
              name="lastName"
              placeholder={t('form_lname')}
              value={formData.lastName}
              onChange={handleInputChange}
              className="flex-1 py-2 px-4 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none h-full min-w-0"
              required
            />
          </div>
          
          {/* Email and Phone Fields */}
          <div className="flex items-start gap-3 w-full h-[43px]">
            <input
              type="email"
              name="email"
              placeholder={t('form_email')}
              value={formData.email}
              onChange={handleInputChange}
              className="flex-1 py-2 px-4 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none h-full min-w-0"
              required
            />
            <input
              type="tel"
              name="phone"
              placeholder={t('form_phone')}
              value={formData.phone}
              onChange={handleInputChange}
              className="flex-1 py-2 px-4 bg-[#E6E6E6] text-[#888] font-roboto text-lg font-light leading-[150%] border-none outline-none h-full min-w-0"
              required
            />
          </div>
          
          {/* Subject Field */}
          <input
            type="text"
            name="subject"
            placeholder={t('form_subject')}
            value={formData.subject}
            onChange={handleInputChange}
            className="flex py-2 px-4 items-center w-full bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none h-[43px]"
            required
          />
          
          {/* Message Field */}
          <textarea
            name="message"
            placeholder={t('form_message')}
            value={formData.message}
            onChange={handleInputChange}
            className="flex py-2 px-4 items-start w-full bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] border-none outline-none resize-none h-[90px]"
            required
          />
          
          {/* Submit Button */}
          <button
            type="submit"
            disabled={status.loading}
            className="flex h-12 py-3 px-6 justify-center items-center gap-2 w-full rounded-sm border border-[#A79D94] bg-white hover:bg-gray-50 transition-colors"
          >
            <span className="text-[#A79D94] font-roboto text-lg font-semibold leading-normal">
              {status.loading ? t('form_sending') : t('form_submit')}
            </span>
          </button>
          {status.error && (
            <p className="text-red-500 text-sm">{status.error}</p>
          )}
          {status.success && (
            <p className="text-green-500 text-sm">{t('form_success_message')}</p>
          )}
        </form>
      </div>

      {/* Social Media Icons - мобильная версия */}
      <div className="flex items-start content-start gap-2 flex-wrap w-full max-w-[351px] h-[88px]">
        {socialIcons.map((icon, index) => (
          <Link 
            key={index}
            href={icon.url}
            target="_blank"
            rel="noopener noreferrer"
            className="w-10 h-10 flex-shrink-0 rounded-full flex justify-center items-center transition-opacity hover:opacity-80"
            style={icon.bgColor ? { backgroundColor: icon.bgColor } : {}}
          >
            <Image
              src={icon.icon}
              alt={icon.name}
              width={40}
              height={40}
              className="flex-shrink-0"
              style={{ maxWidth: '100%', height: 'auto' }}
            />
          </Link>
        ))}
        {/* Social Media Group with text */}
        <div className="flex items-end">
          <Link
            href="https://www.linkedin.com/groups/13198208/"
            target="_blank"
            rel="noopener noreferrer"
            className="w-10 h-10 flex-shrink-0 rounded-full bg-[#42382F] flex justify-center items-center transition-opacity hover:opacity-80"
          >
            <Image
              src="/images/logo/icon_linkedin.svg"
              alt="LinkedIn Group"
              width={40}
              height={40}
              className="flex-shrink-0"
              style={{ maxWidth: '100%', height: 'auto' }}
            />
          </Link>
          <span className="text-[#666] font-manrope text-sm font-normal leading-[150%] tracking-[-0.28px]">
            {t('social_group')}
          </span>
        </div>
      </div>

      {/* Spacer */}
      <div className="h-px w-full max-w-[351px] bg-[rgba(255,255,255,0.1)]"></div>

      {/* Footer Logo and Text - мобильная версия */}
      <div className="flex flex-col justify-center items-start gap-4 w-full max-w-[351px] h-[313px]">
        <div className="w-[65px] h-[65px] flex-shrink-0">
          <Image
            src="/images/logo/logotype_black.svg"
            alt="LAKE Resort Residence"
            width={65}
            height={65}
            className="w-full h-full object-contain"
          />
        </div>
        <p className="w-full text-white font-roboto text-lg font-light leading-[150%]" dangerouslySetInnerHTML={{ __html: t('copyright').replace('Все права защищены.', '<br/>Все права защищены.') }} />
        <p className="w-full text-white font-roboto text-lg font-light leading-[150%]">
          {t('license_info')}
        </p>
      </div>

      {/* Footer Links - мобильная версия */}
      <div className="flex flex-col items-start gap-6 w-full max-w-[351px] h-[66px]">
        <div className="flex items-center gap-6">
          <Link 
            href={getLocalizedPath("/terms-of-use")} 
            className="text-white font-roboto text-lg font-light leading-[150%] hover:opacity-80 transition-opacity"
          >
            {t('terms_of_use')}
          </Link>
          <div className="w-0.5 h-[27px] bg-[rgba(255,255,255,0.1)]"></div>
          <Link 
            href={getLocalizedPath("/privacy-statement")} 
            className="text-white font-roboto text-lg font-light leading-[150%] hover:opacity-80 transition-opacity"
          >
            {t('privacy_statement')}
          </Link>
        </div>
        <Link 
          href={getLocalizedPath("/privacy-settings")} 
          className="text-white font-roboto text-lg font-light leading-[150%] hover:opacity-80 transition-opacity"
        >
          {t('privacy_settings')}
        </Link>
      </div>
    </footer>
  );
} 