'use client';

import { useState, FormEvent, ChangeEvent } from 'react';
import { useTranslations } from 'next-intl';

interface FormData {
  firstName: string;
  lastName: string;
  email: string;
  phone: string;
  subject: string;
  message: string;
}

interface FormStatus {
  loading: boolean;
  success: boolean;
  error: string;
}

export default function ContactForm() {
  const t = useTranslations('InvestorsPage');
  const [formData, setFormData] = useState<FormData>({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    subject: '',
    message: ''
  });

  const [status, setStatus] = useState<FormStatus>({
    loading: false,
    success: false,
    error: ''
  });

  const handleChange = (e: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSubmit = async (e: FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setStatus({ loading: true, success: false, error: '' });

    try {
      const response = await fetch('/api/contact', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(formData),
      });

      if (!response.ok) {
        throw new Error(t('form_error_send_message'));
      }

      setStatus({ loading: false, success: true, error: '' });
      setFormData({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        subject: '',
        message: ''
      });
    } catch (error) {
      setStatus({ loading: false, success: false, error: t('form_error_send_message_try_again') });
    }
  };

  return (
    <form onSubmit={handleSubmit} className="flex w-full md:w-[601px] flex-col justify-center items-center gap-6">
      <h3 className="self-stretch text-white font-roboto text-2xl font-light leading-[130%]">
        {t('form_title')}
      </h3>
      
      <div className="flex h-auto flex-col items-start gap-3 self-stretch">
        {/* Name Fields */}
        <div className="flex flex-col md:flex-row items-start gap-3 self-stretch">
          <input
            type="text"
            name="firstName"
            value={formData.firstName}
            onChange={handleChange}
            placeholder={t('form_fname')}
            required
            className="flex py-2 px-4 items-start gap-[6px] flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] w-full"
          />
          <input
            type="text"
            name="lastName"
            value={formData.lastName}
            onChange={handleChange}
            placeholder={t('form_lname')}
            required
            className="flex py-2 px-4 items-start gap-[6px] flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] w-full"
          />
        </div>
        
        {/* Email and Phone */}
        <div className="flex flex-col md:flex-row items-start gap-3 self-stretch">
          <input
            type="email"
            name="email"
            value={formData.email}
            onChange={handleChange}
            placeholder={t('form_email')}
            required
            className="flex py-2 px-4 items-start gap-[6px] flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] w-full"
          />
          <input
            type="tel"
            name="phone"
            value={formData.phone}
            onChange={handleChange}
            placeholder={t('form_phone')}
            required
            className="flex py-2 px-4 items-start gap-[6px] flex-1 bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] w-full"
          />
        </div>
        
        {/* Subject */}
        <input
          type="text"
          name="subject"
          value={formData.subject}
          onChange={handleChange}
          placeholder={t('form_subject')}
          required
          className="flex py-2 px-4 items-start gap-[6px] self-stretch bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] w-full"
        />
        
        {/* Message */}
        <textarea
          name="message"
          value={formData.message}
          onChange={handleChange}
          placeholder={t('form_format')}
          required
          rows={4}
          className="flex py-2 px-4 items-start gap-[6px] flex-1 self-stretch bg-[#E6E6E6] text-[#666] font-roboto text-lg font-light leading-[150%] w-full resize-none"
        />
      </div>
      
      {status.error && (
        <p className="text-red-500 text-sm">{status.error}</p>
      )}
      
      {status.success && (
        <p className="text-green-500 text-sm">{t('form_success_message')}</p>
      )}
      
      <button
        type="submit"
        disabled={status.loading}
        className="flex h-12 py-3 px-6 justify-center items-center gap-2 self-stretch rounded-[2px] border border-[#A79D94] bg-white disabled:opacity-50"
      >
        <span className="text-[#A79D94] font-roboto text-lg font-semibold leading-normal">
          {status.loading ? t('form_sending') : t('form_button')}
        </span>
      </button>
    </form>
  );
} 