'use client';

import Image from 'next/image';
import { ReactNode, useState } from 'react';
import AnimatedElement from './AnimatedElement';

interface AnimatedImageProps {
  src: string;
  alt: string;
  width?: number;
  height?: number;
  className?: string;
  animation?: 'fadeIn' | 'slideInUp' | 'slideInDown' | 'slideInLeft' | 'slideInRight' | 'zoomIn' | 'bounceIn' | 'flipInX' | 'flipInY' | 'rotateIn';
  hoverEffect?: 'zoom' | 'lift' | 'none';
  delay?: number;
  duration?: number;
  children?: ReactNode;
  priority?: boolean;
  isPanorama?: boolean;
}

export default function AnimatedImage({
  src,
  alt,
  width,
  height,
  className = '',
  animation = 'fadeIn',
  hoverEffect = 'zoom',
  delay = 0,
  duration = 800,
  children,
  priority = false,
  isPanorama = false
}: AnimatedImageProps) {
  const [imageError, setImageError] = useState(false);
  // Не применяем анимации к панорамным изображениям
  if (isPanorama) {
    return (
      <div className={className}>
        {!imageError ? (
          <Image
            src={src}
            alt={alt}
            width={width || 800}
            height={height || 600}
            priority={priority}
            className="w-full h-full object-cover"
            onError={() => setImageError(true)}
          />
        ) : (
          <div className="w-full h-full flex items-center justify-center bg-gray-200 text-gray-500">Ошибка загрузки изображения</div>
        )}
        {children}
      </div>
    );
  }

  const getHoverClass = () => {
    switch (hoverEffect) {
      case 'zoom':
        return 'image-hover-zoom';
      case 'lift':
        return 'image-hover-lift';
      default:
        return '';
    }
  };

  return (
    <AnimatedElement
      animation={animation}
      delay={delay}
      duration={duration}
      className={className}
    >
      <div className={`relative overflow-hidden ${getHoverClass()}`}>
        {!imageError ? (
          <Image
            src={src}
            alt={alt}
            width={width || 800}
            height={height || 600}
            priority={priority}
            className="w-full h-full object-cover"
            onError={() => setImageError(true)}
          />
        ) : (
          <div className="w-full h-full flex items-center justify-center bg-gray-200 text-gray-500">Ошибка загрузки изображения</div>
        )}
        {children}
      </div>
    </AnimatedElement>
  );
} 