'use client';

import { useEffect, useRef, useState, ReactNode } from 'react';

interface AnimatedElementProps {
  children: ReactNode;
  animation?: 'fadeIn' | 'slideInUp' | 'slideInDown' | 'slideInLeft' | 'slideInRight' | 'zoomIn' | 'bounceIn' | 'flipInX' | 'flipInY' | 'rotateIn';
  delay?: number;
  duration?: number;
  threshold?: number;
  className?: string;
  triggerOnce?: boolean;
}

export default function AnimatedElement({
  children,
  animation = 'fadeIn',
  delay = 0,
  duration = 800,
  threshold = 0.1,
  className = '',
  triggerOnce = true
}: AnimatedElementProps) {
  const [isVisible, setIsVisible] = useState(false);
  const [hasAnimated, setHasAnimated] = useState(false);
  const elementRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true);
          if (triggerOnce) {
            setHasAnimated(true);
          }
        } else if (!triggerOnce) {
          setIsVisible(false);
        }
      },
      { threshold }
    );

    if (elementRef.current) {
      observer.observe(elementRef.current);
    }

    return () => observer.disconnect();
  }, [threshold, triggerOnce]);

  const getAnimationClasses = () => {
    const baseClasses = 'transition-all ease-out';
    
    // Проверяем, предпочитает ли пользователь уменьшенное движение
    // Используем безопасную проверку для SSR
    const prefersReducedMotion = typeof window !== 'undefined' 
      ? window.matchMedia('(prefers-reduced-motion: reduce)').matches 
      : false;
    
    if (prefersReducedMotion) {
      return `${baseClasses} opacity-100`;
    }
    
    if (!isVisible) {
      switch (animation) {
        case 'fadeIn':
          return `${baseClasses} opacity-0`;
        case 'slideInUp':
          return `${baseClasses} opacity-0 translate-y-8`;
        case 'slideInDown':
          return `${baseClasses} opacity-0 -translate-y-8`;
        case 'slideInLeft':
          return `${baseClasses} opacity-0 -translate-x-8`;
        case 'slideInRight':
          return `${baseClasses} opacity-0 translate-x-8`;
        case 'zoomIn':
          return `${baseClasses} opacity-0 scale-75`;
        case 'bounceIn':
          return `${baseClasses} opacity-0 scale-50`;
        case 'flipInX':
          return `${baseClasses} opacity-0 rotate-x-90`;
        case 'flipInY':
          return `${baseClasses} opacity-0 rotate-y-90`;
        case 'rotateIn':
          return `${baseClasses} opacity-0 rotate-12 scale-75`;
        default:
          return `${baseClasses} opacity-0`;
      }
    } else {
      return `${baseClasses} opacity-100 translate-y-0 translate-x-0 scale-100 rotate-0`;
    }
  };

  const style = {
    transitionDuration: `${duration}ms`,
    transitionDelay: `${delay}ms`,
  };

  return (
    <div
      ref={elementRef}
      className={`${getAnimationClasses()} ${className}`}
      style={style}
    >
      {children}
    </div>
  );
} 