# Система анимаций LAKE Resort Residence

## Обзор

Система анимаций вдохновлена WordPress Elementor и предоставляет плавные, современные анимации для текстовых блоков и изображений на всех страницах сайта.

## Компоненты

### 1. AnimatedElement
Основной компонент для анимации любых элементов.

```tsx
import AnimatedElement from '../components/AnimatedElement';

<AnimatedElement 
  animation="slideInUp" 
  delay={200} 
  duration={800}
  className="my-class"
>
  <div>Ваш контент</div>
</AnimatedElement>
```

### 2. AnimatedText
Специализированный компонент для анимации текста.

```tsx
import AnimatedText from '../components/AnimatedText';

<AnimatedText
  as="h1"
  animation="slideInUp"
  delay={200}
  className="text-2xl font-bold"
>
  Заголовок
</AnimatedText>
```

### 3. AnimatedImage
Компонент для анимации изображений с эффектами наведения.

```tsx
import AnimatedImage from '../components/AnimatedImage';

<AnimatedImage
  src="/path/to/image.jpg"
  alt="Описание"
  animation="slideInLeft"
  hoverEffect="zoom"
  delay={200}
/>
```

### 4. AnimatedList
Компонент для анимации списков с каскадными задержками.

```tsx
import AnimatedList from '../components/AnimatedList';

<AnimatedList
  animation="slideInUp"
  baseDelay={200}
  staggerDelay={100}
  as="ul"
>
  <li>Элемент 1</li>
  <li>Элемент 2</li>
  <li>Элемент 3</li>
</AnimatedList>
```

### 5. withAnimation HOC
Higher Order Component для оборачивания существующих компонентов.

```tsx
import { withAnimation } from '../components/withAnimation';

const AnimatedButton = withAnimation(Button, {
  animation: 'fadeIn',
  delay: 200
});
```

## Доступные анимации

### Текстовые анимации
- `fadeIn` - плавное появление
- `slideInUp` - появление снизу вверх
- `slideInDown` - появление сверху вниз
- `slideInLeft` - появление слева
- `slideInRight` - появление справа
- `zoomIn` - увеличение при появлении
- `bounceIn` - появление с отскоком
- `flipInX` - переворот по горизонтали
- `flipInY` - переворот по вертикали
- `rotateIn` - появление с поворотом

### Эффекты наведения для изображений
- `zoom` - увеличение при наведении
- `lift` - подъем с тенью при наведении
- `none` - без эффектов

## Применение к существующим страницам

### Главная страница
- ✅ Анимации добавлены к вставным секциям
- ✅ Анимации добавлены к карточкам контента
- ✅ Анимации добавлены к текстовым блокам

### Страница "О нас"
- ✅ Анимации добавлены к заголовкам
- ✅ Анимации добавлены к изображениям
- ✅ Анимации добавлены к текстовым блокам

### Страница "Характеристики"
- ✅ Анимации добавлены к hero секции
- ✅ Анимации добавлены к карточкам характеристик

## Исключения

### Панорамные изображения
Панорамные изображения НЕ анимируются для сохранения стабильности:
- Hero секции
- Фоновые изображения
- Изображения с соотношением сторон < 0.5

## CSS классы

### Автоматические задержки
```css
.stagger-1 { animation-delay: 0.1s; }
.stagger-2 { animation-delay: 0.2s; }
.stagger-3 { animation-delay: 0.3s; }
.stagger-4 { animation-delay: 0.4s; }
.stagger-5 { animation-delay: 0.5s; }
```

### Эффекты наведения
```css
.image-hover-zoom:hover { transform: scale(1.05); }
.image-hover-lift:hover { 
  transform: translateY(-5px);
  box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
}
```

## Рекомендации по использованию

### Для текста
1. Используйте `slideInUp` для заголовков
2. Используйте `fadeIn` для основного текста
3. Добавляйте задержки для создания каскадного эффекта

### Для изображений
1. Используйте `slideInLeft`/`slideInRight` для изображений
2. Применяйте `zoom` эффект для интерактивности
3. Исключайте панорамные изображения

### Для кнопок
1. Используйте `fadeIn` с задержкой
2. Сохраняйте существующие hover эффекты

## Производительность и доступность

- Анимации используют CSS transitions для оптимальной производительности
- Intersection Observer используется для запуска анимаций при появлении в viewport
- Анимации автоматически отключаются на устройствах с `prefers-reduced-motion`
- Поддержка WCAG 2.1 для пользователей с нарушениями вестибулярного аппарата
- Плавные переходы без резких движений

## Добавление анимаций к новым страницам

1. Импортируйте необходимые компоненты:
```tsx
import AnimatedText from '../components/AnimatedText';
import AnimatedImage from '../components/AnimatedImage';
import AnimatedElement from '../components/AnimatedElement';
```

2. Замените обычные элементы на анимированные:
```tsx
// Было
<h1>Заголовок</h1>

// Стало
<AnimatedText as="h1" animation="slideInUp" delay={200}>
  Заголовок
</AnimatedText>
```

3. Для изображений:
```tsx
// Было
<img src="/image.jpg" alt="Описание" />

// Стало
<AnimatedImage 
  src="/image.jpg" 
  alt="Описание"
  animation="slideInLeft"
  hoverEffect="zoom"
/>
``` 